/* wincap.h: Header for OS capability class.

This file is part of Cygwin.

This software is a copyrighted work licensed under the terms of the
Cygwin license.  Please consult the file "CYGWIN_LICENSE" for
details. */

#ifndef _WINCAP_H
#define _WINCAP_H

#include "memory_layout.h"

struct wincaps
{
  /* The bitfields must be 8 byte aligned on x86_64, otherwise the bitfield
     ops generated by gcc are off by 4 bytes. */
  struct  __attribute__ ((aligned (8))) {
    unsigned has_new_pebteb_region				: 1;
    unsigned has_unprivileged_createsymlink			: 1;
    unsigned has_precise_interrupt_time				: 1;
    unsigned has_posix_unlink_semantics				: 1;
    unsigned has_posix_unlink_semantics_with_ignore_readonly	: 1;
    unsigned has_case_sensitive_dirs				: 1;
    unsigned has_posix_rename_semantics				: 1;
    unsigned has_con_24bit_colors				: 1;
    unsigned has_con_broken_csi3j				: 1;
    unsigned has_con_broken_il_dl				: 1;
    unsigned has_con_esc_rep					: 1;
    unsigned has_extended_mem_api				: 1;
    unsigned has_tcp_fastopen					: 1;
    unsigned has_linux_tcp_keepalive_sockopts			: 1;
    unsigned has_tcp_maxrtms					: 1;
    unsigned has_con_broken_tabs				: 1;
    unsigned has_user_shstk					: 1;
    unsigned has_pcon_omit_nl_before_cursor_move		: 1;
  };
};

class wincapc
{
  SYSTEM_INFO		system_info;
  RTL_OSVERSIONINFOEXW	version;
  char			osnam[40];
  const void		*caps;
  USHORT		host_mach;
  USHORT		cygwin_mach;
  bool			_is_server;

public:
  void init ();

  const DWORD cpu_count () const { return system_info.dwNumberOfProcessors; }
  const DWORD_PTR cpu_mask () const { return system_info.dwActiveProcessorMask;}

  const WORD cpu_arch () const { return system_info.wProcessorArchitecture; }
  const WORD cpu_level () const { return system_info.wProcessorLevel; }

  /* The casts to size_t make sure that the returned value has the size of
     a pointer on any system.  This is important when using them for bit
     mask operations, like in roundup2. */
  const size_t page_size () const { return (size_t) system_info.dwPageSize; }
  const size_t allocation_granularity () const
		     { return (size_t) system_info.dwAllocationGranularity; }
  const char *osname () const { return osnam; }
  const DWORD build_number () const { return version.dwBuildNumber; }
  const USHORT host_machine () const { return host_mach; }
  const USHORT cygwin_machine () const { return cygwin_mach; }

#define IMPLEMENT(cap) cap() const { return ((wincaps *) this->caps)->cap; }

  DWORD def_guard_page_size () const
  {
    return DEFAULT_GUARD_PAGE_COUNT * page_size ();
  }
  bool  is_server () const { return _is_server; }
  bool	IMPLEMENT (has_new_pebteb_region)
  bool	IMPLEMENT (has_unprivileged_createsymlink)
  bool	IMPLEMENT (has_precise_interrupt_time)
  bool	IMPLEMENT (has_posix_unlink_semantics)
  bool	IMPLEMENT (has_posix_unlink_semantics_with_ignore_readonly)
  bool	IMPLEMENT (has_case_sensitive_dirs)
  bool	IMPLEMENT (has_posix_rename_semantics)
  bool	IMPLEMENT (has_con_24bit_colors)
  bool	IMPLEMENT (has_con_broken_csi3j)
  bool	IMPLEMENT (has_con_broken_il_dl)
  bool	IMPLEMENT (has_con_esc_rep)
  bool	IMPLEMENT (has_extended_mem_api)
  bool	IMPLEMENT (has_tcp_fastopen)
  bool	IMPLEMENT (has_linux_tcp_keepalive_sockopts)
  bool	IMPLEMENT (has_tcp_maxrtms)
  bool	IMPLEMENT (has_con_broken_tabs)
  bool	IMPLEMENT (has_user_shstk)
  bool	IMPLEMENT (has_pcon_omit_nl_before_cursor_move)

  void disable_case_sensitive_dirs ()
  {
    ((wincaps *)caps)->has_case_sensitive_dirs = false;
  }
#undef IMPLEMENT
};

extern wincapc wincap;

#endif /* _WINCAP_H */
